/* 
 * Author: Mark Thomas 
 * Lexical Analyser written using Lex
 */

/* denotes a letter */
L                       [a-zA-Z_]

/* denotes a digit */
D                       [0-9]

/* denotes whitespace(s) */
W                       [ \t\v\n\f]

/* denotes end of line character */
N                       [\n]

%{
#include <stdio.h>
#include "node.h"
#include "parse.tab.h"
#include "xalloc.h"

/* global variables as used in Lex */

extern YYSTYPE yylval;
extern int yydebug;

/* function prototypes */

int yywrap(void);
void yyerror(char *);
void count(void);
int checkType(void);
int comment(void);

/* stores current line number */
static int sLineNumber = 1;
static int column = 0;
%}


/* List of Tokens that the Lex uses */

%%

"/*"                    { count();      comment(); }
"break"                 { count();      yylval.sValue = BREAK;          return(BREAK); }
"continue"              { count();      yylval.sValue = CONTINUE;       return(CONTINUE); }
"return"                { count();      yylval.sValue = RETURN;         return(RETURN); }
"while"                 { count();      yylval.sValue = WHILE;          return(WHILE); }
"else"                  { count();      yylval.sValue = ELSE;           return(ELSE); }
"if"                    { count();      yylval.sValue = IF;             return(IF); } 
"int"                   { count();      yylval.sValue = INT;            return(INT); }

{N}                     { count();      sLineNumber++; }
{L}({L}|{D})*           { count();      return checkType(); }
{D}({D})*               { count();      yylval.sValue = atoi(yytext);   return(INT_DENOTATION); }
{W}                     { count();      yylval.sValue = 0; }

"+"                     { count();      yylval.sValue = '+';            return('+'); }
"-"                     { count();      yylval.sValue = '-';            return('-'); }
"*"                     { count();      yylval.sValue = '*';            return('*'); }
"/"                     { count();      yylval.sValue = '/';            return('/'); }
"%"                     { count();      yylval.sValue = '%';            return('%'); }
"!"                     { count();      yylval.sValue = '!';            return('!'); }
"?"                     { count();      yylval.sValue = '?';            return('?'); }
":"                     { count();      yylval.sValue = ':';            return(':'); }
"="                     { count();      yylval.sValue = '=';            return('='); }
","                     { count();      yylval.sValue = ',';            return(','); }
">"                     { count();      yylval.sValue = '>';            return('>'); }
"<"                     { count();      yylval.sValue = '<';		return('<'); }
"("                     { count();      yylval.sValue = 0;              return('('); }
")"                     { count();      yylval.sValue = 0;              return(')'); }
"{"                     { count();      yylval.sValue = 0;              return('{'); }
"}"                     { count();      yylval.sValue = 0;              return('}'); }
"||"                    { count();      yylval.sValue = OR_OP;          return(OR_OP); }
"&&"                    { count();      yylval.sValue = AND_OP;         return(AND_OP); }
"=="                    { count();      yylval.sValue = EQ_OP;          return(EQ_OP); }
";"                     { count();      yylval.sValue = 0;              return(';'); }

.                       { yyerror("Bad character"); exit(1);}

%%

/* Code section */


/* use only one lex file for analysis */

int
yywrap(void)
{
        return(1);
}


/* copy identifier value and return it */

int 
checkType(void)
{
	yylval.pszValue = xstrdup(yytext);	/* copy the string */

   	return(IDENTIFIER);
}


/* ignore comments */

int
comment(void)
{
    int c;

    for (;;){	
	/* read each character until we find close comment */
        while ((c = input()) != '*' && c != EOF)
                if (c == '\n') sLineNumber++;

                if (c == '*'){
                    while ((c = input()) == '*') 
                        ;
                    if (c == '\n') sLineNumber++;
                    if (c == '/') break;
                }

                if(c == EOF)
                    printf("unterminated comment\n");
        }
}

/* calculate the current position on the line */

void count(void)
{
        int i;
 
        for(i = 0; yytext[i] != '\0'; i++)
                if(yytext[i] == '\n')
                        column = 0;
                else if(yytext[i] == '\t')           
                        column += 8 - (column % 8);
                else
                        column++;
         
        ECHO;
}

/* report an error to the standard error stream */

void
yyerror( char *s )
{
    	fflush(stdout);
    	fprintf(stderr, "\n%*s\n%*s at position %d on line %d\n",
            column, "^", column, s, column, sLineNumber);
}
