#include <stdio.h>
#include <assert.h>
#include <string.h>
#include "xalloc.h"
#include "node.h"
#include "nameAnalyser.h"

void
initEnvNode(EnvNode * node) {
	node->firstChild = NULL;
    	node->lastChild = NULL;
    	node->firstIdent = NULL;
    	node->lastIdent = NULL;
    	node->parent = NULL;
    	node->nextSibling = NULL;
}

void
addEnvChild(EnvNode * parent) {
    	EnvNode * newNode = XMALLOC(EnvNode , 1);
    
    	assert(parent != NULL);

    	initEnvNode(newNode);
    	newNode->parent = parent;

    	if(parent->firstChild == NULL){
        	parent->firstChild = newNode;
        	parent->lastChild = newNode;
    	}
    	else {
        	parent->lastChild->nextSibling = newNode;
        	parent->lastChild = newNode;
    	}
}

void
addEnvIdent(EnvNode * node, char * name, SymbolTable * symTab) {
    	Identifier * newIdent;
    
    	assert(node != NULL && name != NULL && symTab != NULL);
    	newIdent = XMALLOC(Identifier, 1);
    	newIdent->name = XMALLOC(char, strlen(name)+1);
    	strcpy(newIdent->name, name);
   	newIdent->index = addSymTabEntry(symTab);

   	if(node->firstIdent == NULL) {
        	node->firstIdent = newIdent;
        	node->lastIdent = newIdent;
    	}
    	else {
        	node->lastIdent->next = newIdent;
        	node->lastIdent = newIdent;
    	}
}

int
getIdentIndex(EnvNode * node, char * name) {
    	Identifier * currIdent;

    	assert(name != NULL);
    
    	if(node == NULL)
        	return -1;

    	for(currIdent = node->firstIdent;
        	currIdent != NULL;
        	currIdent = currIdent->next)
	{
        	if(strncmp(currIdent->name, name, MAX_VAR_NAME_SIZE) == 0)
            		return currIdent->index;
    	}

    		return getIdentIndex(node->parent, name);
}

int
isInEnv(EnvNode * node, char * name) {
    	Identifier * currIdent;
    
    	assert(node != NULL && name != NULL);

    	for(currIdent = node->firstIdent; 
		currIdent != NULL; 
		currIdent = currIdent->next) 
	{
        	if(strncmp(currIdent->name, name, MAX_VAR_NAME_SIZE) == 0)
            		return 1;
    	}

    	return 0;

}

void
deleteEnvNode(EnvNode * node) {
    	Identifier * currIdent;
    	Identifier * prevIdent;
    	EnvNode * currChild;
    	EnvNode * prevChild;

    	assert(node != NULL);

    	for(currIdent = node->firstIdent; currIdent != NULL; ){
        	deleteIdent(currIdent);
        	prevIdent = currIdent;
        	currIdent = currIdent->next;

        	XFREE(prevIdent);
    	}

    	for(currChild = node->firstChild; currChild != NULL; ){
        	deleteEnvNode(currChild);
        	prevChild = currChild;
        	currChild = currChild->nextSibling;
        
        	XFREE(prevChild);
    	}
}

void
deleteIdent(Identifier * ident){
    	XFREE(ident->name);

}

void
initSymTab(SymbolTable * symTab){
    	assert(symTab != NULL);
    
    	symTab->entries = XMALLOC(STEntry, DEFAULT_ST_SIZE);
    	symTab->numEntries = 0;
    	symTab->size =  DEFAULT_ST_SIZE;
}


/* returns the index of the new entry */

int
addSymTabEntry(SymbolTable * symTab) {
    	STEntry * newEntries;
    	int i;
        
    	assert(symTab != NULL);

    	/* resize the table if necessary */
 
   	if(symTab->numEntries == symTab->size) {
        	newEntries = XCALLOC(STEntry, 2*(symTab->size));
		memcpy(newEntries, symTab->entries, sizeof(STEntry)*(symTab->size));
        	XFREE(symTab->entries);
        	symTab->entries = newEntries;
        	symTab->size = 2*(symTab->size);
    	}
    
    	symTab->numEntries++;

    	/* return the index of the new entry */
    	return(symTab->numEntries - 1);

}

void
deleteSymTab(SymbolTable * symTab) {
    	XFREE(symTab->entries);
}


int
analyseTree(NODE * root, EnvNode * rootEnv, SymbolTable * symTab) {
    	NODE * currNode;
    	char* identName;
    	int identIndex;
    	int errorFound = 0;
    
    	if(root == NULL)
        	return;

    	switch(root->sNodeType) {
    	case NODE_DECLARATION:
 
       /* left node will be an identifier */

        identName = root->NodeLeft->vari.szName;

        if(!isInEnv(rootEnv, identName))
            	addEnvIdent(rootEnv, identName, symTab);

        else {
            errorFound = 1;
            printf("Error: Identifier %s already declared. Line: %i.\n", identName, root->sLineNumber);
            break;	
	}

        errorFound = analyseTree(root->NodeLeft, rootEnv, symTab);

        if(root->NodeRight != NULL)
            	errorFound = analyseTree(root->NodeRight, rootEnv, symTab);
        break;

    	case NODE_VARIABLE:
        	identIndex = getIdentIndex(rootEnv, root->vari.szName);
        	if(identIndex != -1)
            		root->sTableIndex = identIndex;
        	else {
            		errorFound = 1;
            		printf("Error: Identifier %s undeclared. Line: %i\n",  root->vari.szName,
	    		root->sLineNumber);	
        	}
        break;

        /* nodes that contain nothing to check */

    	case NODE_BREAK:
    	case NODE_CONTINUE:
        	break;
    	case NODE_NUMBER:
        	break;
        
        /* unary nodes */

    	case NODE_RETURN:
    	case NODE_NEG_OP:
    	case NODE_NOT_OP:
        	errorFound = analyseTree(root->NodeLeft, rootEnv, symTab);
        	break;

    	case NODE_BLOCK:
        /* blocks require a new scope */
        	addEnvChild(rootEnv);
        	rootEnv = rootEnv->lastChild;

        	for(currNode = root->NodeLeft; 
			currNode != NULL && !errorFound; 
			currNode = currNode->NodeNext)

            		errorFound = analyseTree(currNode, rootEnv, symTab);

        	for(currNode = root->NodeRight; 
			currNode !=NULL && !errorFound; 
			currNode = currNode->NodeNext)

            			errorFound = analyseTree(currNode, rootEnv, symTab);
        	break;

    	case NODE_PROGRAM:
		for(currNode = root->NodeLeft; 
			currNode != NULL && !errorFound; 
			currNode = currNode->NodeNext) 
		{
	    		errorFound = analyseTree(currNode, rootEnv, symTab);
	    		symTab->entries[currNode->NodeLeft->sTableIndex].isParam = 1;
		}

        	errorFound = analyseTree(root->NodeRight, rootEnv, symTab);
        	break;
    
        /* binary nodes */
    	default:
        	errorFound = analyseTree(root->NodeLeft, rootEnv, symTab);
        	errorFound = analyseTree(root->NodeRight, rootEnv, symTab);
    	}
    
    	if (errorFound) {
    		deleteSymTab(symTab);
		exit(0);
	}

    	return errorFound;
}
